#ifndef _TVR_BASE_16_STOP_THE_INSANITY_H
#define _TVR_BASE_16_STOP_THE_INSANITY_H

/*
This code is released into the public domain by Joseph E. Van Riper III.

NOTE:
To use this, simply include the header and write your code.  This requires no static or dynamic library
files short of whatever might come with your standard C++ library.  You may need to ensure you have an
'unordered_map' in your standard C++ library, as this is a relatively recent addition to the standard.
 */

#include <string>
#include <sstream>
#include <unordered_map>

namespace tvr
{
	namespace encode
	{
		/**
		 * base16 encoder
		 * Encodes arbitrary binary files to an alphabet of 16 characters per RFC 4648, with some extra
		 * little features to make this somewhat flexible.
		 **/
		class base_16
		{
		public:
			/// This is the normal alphabet for the base16 encoding, per RFC 4648.
			static const std::string& get_standard_alphabet()
			{
				static const std::string alphabet( "0123456789ABCDEF" );
				return alphabet;
			}
			/// This is the usual set of characters used to end a line.
			static const std::string& get_standard_line_end()
			{
#ifdef WIN32
				static const std::string line_end( "\r\n" );
#else
				static const std::string line_end( "\n" );
#endif
				return line_end;
			}
		public:
			/// Constructor with normal defaults.
			/// 'alphabet' sets the alphabet used for generating the encoded results.
			///            You are responsible for maintaining its memory for the life
			///            of this object.
			/// 'chars_per_line' sets the number of encoded characters to set per line.
			/// 'line_end' sets the characters used at the end of the line to mark the line's end.
			base_16( const std::string& alphabet = get_standard_alphabet(),
				unsigned int chars_per_line = 76,
				const std::string& line_end = get_standard_line_end() ):
			  _alphabet( alphabet ),
			  _line_end( line_end ),
			  _chars_per_line( chars_per_line )
			{
			}

			/// Destructor.
			~base_16()
			{
			}

			/// Encode the incoming buffer and reset the object as fresh for use.
			std::string encode_full( const std::string& incoming )
			{
				std::string result;
				result.resize( ( incoming.size() * 2 ) + ( ( ( incoming.size() * 2 ) / _line_end.size() ) + 1 ) );
				unsigned int line_count = 0;
				std::string::size_type result_pos = 0;
				for ( std::string::const_iterator iter = incoming.begin(); iter != incoming.end(); ++iter )
				{
					result[ result_pos++ ] = encode_value( ( *iter & 0x0f0 ) >> 4 );
					result[ result_pos++ ] = encode_value( *iter & 0x00f );
					if ( _chars_per_line != 0 )
					{
						line_count += 2;
						if ( line_count + 2 > _chars_per_line )
						{
							for ( std::string::const_iterator pos = _line_end.begin(); pos != _line_end.end(); ++pos )
							{
								result[ result_pos++ ] = *pos;
							}
							line_count = 0;
						}
					}
				}
				return result.c_str();
			}

			/// Stream version of the encoder, reading 'in' until it has nothing else to provide, and
			/// streaming its results to 'out'.  Object is reset as fresh for use again when finished.
			std::ostream& encode( std::ostream& out, std::istream& in, unsigned int buffer_size = 4096 )
			{
				std::string plaintext;
				plaintext.resize( buffer_size );
				std::streamsize plainlength;

				do
				{
					in.read( &plaintext[0], plaintext.size() );
					plainlength = in.gcount();
					if ( plainlength > 0 )
					{
						std::string in_text( &plaintext[0], &plaintext[0] + plainlength );
						std::string result = encode_full( in_text );
						out.write( &result[0], result.size() );
					}
				}
				while (in.good() && plainlength > 0);

				return out;
			}

		private:

			char encode_value( char in )
			{
				return _alphabet[ ( in & 0x00f ) ];
			}
		private:
			const std::string& _alphabet;
			const std::string& _line_end;
			unsigned int _chars_per_line;
		}; // class base_16
	}; // namespace encode

	namespace decode
	{
		/**
		 * base64 decoder
		 * Decodes from an alphabet of 16 characters per RFC 4648 to binary, with some extra
		 * little features to make this somewhat flexible.
		 **/
		class base_16
		{
		private:
			typedef std::unordered_map< char, char > char_lookup_t;
		public:
			/// The standard alphabet from which to decode the cypher.
			static const std::string& get_standard_alphabet()
			{
				return encode::base_16::get_standard_alphabet();
			}
		private:
			
			enum e_state
			{
				step_A,
				step_B
			};

			struct state
			{
				e_state _step;
				char plainchar;
			};
		public:
			/// Constructor that takes a previously-agreed-upon alphabet.  The alphabet
			/// used here is presumed to have been used for the encoder object earlier.
			base_16( const std::string& alphabet = get_standard_alphabet() ):
			  _alphabet( alphabet )
			{
				build_map();
				init();
			}

			void init()
			{
				_state.plainchar = 0;
				_state._step = step_A;
			}

			/// Decodes the text with the assumption that this is all of the text.  It resets
			/// this object for use for a different data stream when finished.
			std::string decode_full( const std::string& in )
			{
				std::string result;
				decode_block( result, in );
				init();
				return result.c_str();
			}

			/// Decodes a chunk of the text.  Call 'decode_full' for the last chunk, or 'init' when finished.
			/// Should the other half of the hexidecimal character not be available by the end, the first of
			// the pair is thrown away.
			std::string decode_partial( const std::string& in )
			{
				std::string result;
				decode_block( result, in );
				return result.c_str();
			}

			/// Decodes from a stream to a stream, like the encoder's version.  When finished,
			/// it resets this object for use for a different data stream.
			std::ostream& decode( std::ostream& out, std::istream& in, unsigned int buffer_size = 4096 )
			{
				std::string plaintext;
				plaintext.resize( buffer_size );
				std::streamsize plainlength;

				do
				{
					in.read( &plaintext[0], plaintext.size() );
					plainlength = in.gcount();
					if ( plainlength > 0 )
					{
						std::string in_text( &plaintext[0], &plaintext[0] + plainlength );
						std::string result = decode_full( in_text );
						out.write( &result[0], result.size() );
					}
				}
				while (in.good() && plainlength > 0);
				return out;
			}

		private:

			int decode_value( char in )
			{
				int result = -1;
				if ( _alphabet_map.empty() )
				{
					build_map();
				}
				char_lookup_t::const_iterator found = _alphabet_map.find( in );
				if ( found != _alphabet_map.end() )
				{
					result = found->second;
				}
				return result;
			}

			int decode_block( std::string& out, const std::string& in )
			{
				out.resize( ( in.size() / 2 ) + 1 );

				std::string::size_type out_pos = 0;
				std::string::const_iterator in_iter = in.begin();
				int fragment = 0;
				out[ out_pos ] = _state.plainchar;

				switch ( _state._step )
				{
					while ( 1 )
					{
				case step_A:
						do {
							if ( in_iter == in.end() )
							{
								_state._step = step_A;
								_state.plainchar = out[ out_pos ];
								return out_pos;
							}
							fragment = (char)decode_value( *in_iter++ );
						}
						while ( fragment < 0 );
						out[ out_pos ] = ( fragment & 0x00f ) << 4;
				case step_B:
						do {
							if ( in_iter == in.end() )
							{
								_state._step = step_B;
								_state.plainchar = out[ out_pos ];
								return out_pos;
							}
							fragment = (char)decode_value( *in_iter++ );
						}
						while ( fragment < 0 );
						out[ out_pos++ ] |= ( fragment & 0x00f );
					}
				}
				return out_pos;
			}

			void build_map()
			{
				char ch = 0;
				for ( std::string::const_iterator iter = _alphabet.begin();
					iter != _alphabet.end();
					++iter )
				{
					_alphabet_map[ *iter ] = ch++;
				}
			}
		private:
			const std::string& _alphabet;
			char_lookup_t _alphabet_map;
			state _state;
		}; // class base_16
	}; // namespace decode
}; // namespace tvr

#endif // _TVR_BASE_16_STOP_THE_INSANITY_H
