#ifndef _TVR_BASE_32_STOP_THE_INSANITY_H
#define _TVR_BASE_32_STOP_THE_INSANITY_H

/*
This code is released into the public domain by Joseph E. Van Riper III.  The heart of this code comes
from http://sourceforge.net/projects/libb64/, which is also released into the public domain.  My
contributions involve making this more C++-friendly, adding a few extra features to make it a
little more flexible than the original code, and encoding for base32 instead of base64.

NOTE:
To use this, simply include the header and write your code.  This requires no static or dynamic library
files short of whatever might come with your standard C++ library.  You may need to ensure you have an
'unordered_map' in your standard C++ library, as this is a relatively recent addition to the standard.
 */

#include <string>
#include <sstream>
#include <unordered_map>

namespace tvr
{
	namespace encode
	{
		/**
		 * base32 encoder
		 * Encodes arbitrary binary files to an alphabet of 32 characters per RFC 4648, with some extra
		 * little features to make this somewhat flexible.
		 **/
		class base_32
		{
		private:
			enum e_state
			{
				step_A,
				step_B,
				step_C,
				step_D,
				step_E
			};
			struct state
			{
				e_state _step;
				char _result;
				int _stepcount;
			};
		public:
			/// This is the normal alphabet for the base32 encoding, per RFC 4648.
			static const std::string& get_standard_alphabet()
			{
				static const std::string alphabet( "ABCDEFGHIJKLMNOPQRSTUVWXYZ234567" );
				return alphabet;
			}
			/// This is the extended hex alphabet mentioned in RFC 4648 (base32hex).
			static const std::string& get_extended_hex_alphabet()
			{
				static const std::string alphabet( "0123456789ABCDEFGHIJKLMNOPQRSTUV" );
				return alphabet;
			}
			/// This is the usual set of characters used to end a line.
			static const std::string& get_standard_line_end()
			{
#ifdef WIN32
				static const std::string line_end( "\r\n" );
#else
				static const std::string line_end( "\n" );
#endif
				return line_end;
			}
		public:
			/// Constructor with normal defaults.
			/// 'alphabet' sets the alphabet used for generating the encoded results.
			///            You are responsible for maintaining its memory for the life
			///            of this object.
			/// 'chars_per_line' sets the number of encoded characters to set per line.
			/// 'line_end' sets the characters used at the end of the line to mark the line's end.
			/// 'pad' sets the character used as the pad character when the bytes do not align.
			base_32( const std::string& alphabet = get_standard_alphabet(),
				unsigned int chars_per_line = 76,
				const std::string& line_end = get_standard_line_end(),
				const char pad = '=' ):
			  _alphabet( alphabet ),
			  _pad( pad ),
			  _line_end( line_end ),
			  _chars_per_line( chars_per_line )
			{
				init();
			}

			/// Destructor.
			~base_32()
			{
			}

			/// Resets the object as if no bytes have passed through it.
			void init()
			{
				_state._step = step_A;
				_state._result = 0;
				_state._stepcount = 0;
			}

			/// Encode the incoming buffer and reset the object as fresh for use.
			std::string encode_full( const std::string& incoming )
			{
				std::string result, ending;
				result = encode_partial( incoming );
				ending = encode_finish();
				std::ostringstream strm;
				strm << result.c_str() << ending.c_str();
				return strm.str();
			}

			/// Encode the incoming buffer, but do not reset it.  Use this when you get your data in batches,
			/// and concatenate the results.  Call 'encode_finish' when you have no more data, and concatenate
			/// its results to these results.
			std::string encode_partial( const std::string& incoming )
			{
				std::string result;
				encode_block( result, incoming );
				return result.c_str();
			}

			/// Cleans out any leftover bytes, provides padding chars if required, and resets the object as
			/// fresh for use again.
			std::string encode_finish()
			{
				std::string result;
				encode_block_end( result );
				init();
				return result.c_str();
			}

			/// Stream version of the encoder, reading 'in' until it has nothing else to provide, and
			/// streaming its results to 'out'.  Object is reset as fresh for use again when finished.
			std::ostream& encode( std::ostream& out, std::istream& in, unsigned int buffer_size = 4096 )
			{
				std::string plaintext;
				plaintext.resize( buffer_size );
				std::streamsize plainlength;

				init();

				do
				{
					in.read( &plaintext[0], plaintext.size() );
					plainlength = in.gcount();
					if ( plainlength > 0 )
					{
						std::string in_text( &plaintext[0], &plaintext[0] + plainlength );
						std::string result = encode_partial( in_text );
						out.write( &result[0], result.size() );
					}
				}
				while (in.good() && plainlength > 0);

				std::string result = encode_finish();
				out.write( &result[0], result.size() );

				init();
				return out;
			}

		private:

			int encode_block( std::string& out, const std::string& in )
			{
				std::string::const_iterator iter = in.begin();
				out.resize( ( ( ( in.size() / 3 ) + 1 ) * 9 ) );
				std::string::size_type out_pos = 0;
				char fragment = 0;
				char result = 0;

				switch( _state._step )
				{
					while ( 1 )
					{
				case step_A:
						if ( iter == in.end() )
						{
							_state._result = result;
							_state._step = step_A;
							return out_pos;
						}
						fragment = *iter++;
						result = ( fragment & 0x0f8 ) >> 3;
						out[ out_pos++ ] = encode_value( result );
						result = ( fragment & 0x007 ) << 2;
				case step_B:
						if ( iter == in.end() )
						{
							_state._result = result;
							_state._step = step_B;
							return out_pos;
						}
						fragment = *iter++;
						result |= ( fragment & 0x0c0 ) >> 6;
						out[ out_pos++ ] = encode_value( result );
						result = ( fragment & 0x03e ) >> 1;
						out[ out_pos++ ] = encode_value( result );
						result = ( fragment & 0x001 ) << 4;
				case step_C:
						if ( iter == in.end() )
						{
							_state._result = result;
							_state._step = step_C;
							return out_pos;
						}
						fragment = *iter++;
						result |= ( fragment & 0x0f0 ) >> 4;
						out[ out_pos++ ] = encode_value( result );
						result = ( fragment & 0x00f ) << 1;
				case step_D:
						if ( iter == in.end() )
						{
							_state._result = result;
							_state._step = step_D;
							return out_pos;
						}
						fragment = *iter++;
						result |= ( fragment & 0x080 ) >> 7;
						out[ out_pos++ ] = encode_value( result );
						result = ( fragment & 0x07c ) >> 2;
						out[ out_pos++ ] = encode_value( result );
						result = ( fragment & 0x003 ) << 3;
				case step_E:
						if ( iter == in.end() )
						{
							_state._result = result;
							_state._step = step_E;
							return out_pos;
						}
						fragment = *iter++;
						result |= ( fragment & 0x0e0 ) >> 5;
						out[ out_pos++ ] = encode_value( result );
						result = ( fragment & 0x01f );
						out[ out_pos++ ] = encode_value( result );

						++_state._stepcount;
						if ( _state._stepcount == _chars_per_line / 8 )
						{
							for ( std::string::const_iterator pos = _line_end.begin(); pos != _line_end.end(); ++pos )
							{
								out[ out_pos++ ] = *pos;
							}
							_state._stepcount = 0;
						}
					}
				}
				return out_pos;
			}

			int encode_block_end( std::string& out )
			{
				std::string::size_type out_pos = 0;
				out.resize( 8 + _line_end.size() );

				switch ( _state._step )
				{
				case step_B:
					out[ out_pos++ ] = encode_value( _state._result );
					out[ out_pos++ ] = _pad;
					out[ out_pos++ ] = _pad;
					out[ out_pos++ ] = _pad;
					out[ out_pos++ ] = _pad;
					out[ out_pos++ ] = _pad;
					out[ out_pos++ ] = _pad;
					break;
				case step_C:
					out[ out_pos++ ] = encode_value( _state._result );
					out[ out_pos++ ] = _pad;
					out[ out_pos++ ] = _pad;
					out[ out_pos++ ] = _pad;
					out[ out_pos++ ] = _pad;
					break;
				case step_D:
					out[ out_pos++ ] = encode_value( _state._result );
					out[ out_pos++ ] = _pad;
					out[ out_pos++ ] = _pad;
					out[ out_pos++ ] = _pad;
				case step_E:
					out[ out_pos++ ] = encode_value( _state._result );
					out[ out_pos++ ] = _pad;
				case step_A:
					break;
				}

				for ( std::string::const_iterator pos = _line_end.begin(); pos != _line_end.end(); ++pos )
				{
					out[ out_pos++ ] = *pos;
				}

				return out_pos;
			}

			char encode_value( char in )
			{
				if ( in > 63 )
				{
					return _pad;
				}
				return _alphabet[ in ];
			}
		private:
			state _state;
			const std::string& _alphabet;
			const std::string& _line_end;
			const char _pad;
			unsigned int _chars_per_line;
		}; // class base_32
	}; // namespace encode

	namespace decode
	{
		/**
		 * base32 decoder
		 * Decodes from an alphabet of 32 characters per RFC 4648 to binary, with some extra
		 * little features to make this somewhat flexible.
		 **/
		class base_32
		{
		private:
			typedef std::unordered_map< char, char > char_lookup_t;
			enum e_state
			{
				step_A,
				step_B,
				step_C,
				step_D,
				step_E,
				step_F,
				step_G,
				step_H
			};

			struct state
			{
				e_state _step;
				char plainchar;
			};
		public:
			/// The standard alphabet from which to decode the cypher.
			static const std::string& get_standard_alphabet()
			{
				return encode::base_32::get_standard_alphabet();
			}
			/// The base32hex alphabet (from RFC 4648).
			static const std::string& get_extended_hex_alphabet()
			{
				return encode::base_32::get_extended_hex_alphabet();
			}
		public:
			/// Constructor that takes a previously-agreed-upon alphabet.  The alphabet
			/// used here is presumed to have been used for the encoder object earlier.
			base_32( const std::string& alphabet = get_standard_alphabet() ):
			  _alphabet( alphabet )
			{
				build_map();
				init();
			}

			/// Resets the object for fresh use again.
			void init()
			{
				_state._step = step_A;
				_state.plainchar = 0;
			}

			/// Decodes the text, with the assumption that more data is coming.  Use this
			/// to decode batches of the full data at a time.  When finished, call init to
			/// reset this object for a different data stream.
			std::string decode_partial( const std::string& in )
			{
				std::string result;
				decode_block( result, in );
				return result.c_str();
			}

			/// Decodes the text with the assumption that this is all of the text.  It resets
			/// this object for use for a different data stream when finished.
			std::string decode_full( const std::string& in )
			{
				std::string result;
				decode_block( result, in );
				init();
				return result.c_str();
			}

			/// Decodes from a stream to a stream, like the encoder's version.  When finished,
			/// it resets this object for use for a different data stream.
			std::ostream& decode( std::ostream& out, std::istream& in, unsigned int buffer_size = 4096 )
			{
				std::string plaintext;
				plaintext.resize( buffer_size );
				std::streamsize plainlength;

				init();

				do
				{
					in.read( &plaintext[0], plaintext.size() );
					plainlength = in.gcount();
					if ( plainlength > 0 )
					{
						std::string in_text( &plaintext[0], &plaintext[0] + plainlength );
						std::string result = decode_partial( in_text );
						out.write( &result[0], result.size() );
					}
				}
				while (in.good() && plainlength > 0);

				init();
				return out;
			}

		private:
			int decode_block( std::string& out, const std::string& in )
			{
				if ( out.size() < in.size() )
				{
					out.resize( in.size() );
				}

				std::string::size_type out_pos = 0;
				std::string::const_iterator in_iter = in.begin();
				int fragment = 0;
				out[ out_pos ] = _state.plainchar;

				switch ( _state._step )
				{
					while ( 1 )
					{
				case step_A:
						do {
							if ( in_iter == in.end() )
							{
								_state._step = step_A;
								_state.plainchar = out[ out_pos ];
								return out_pos;
							}
							fragment = (char)decode_value( *in_iter++ );
						}
						while ( fragment < 0 );
						out[ out_pos ] = ( fragment & 0x01f ) << 3;
				case step_B:
						do {
							if ( in_iter == in.end() )
							{
								_state._step = step_B;
								_state.plainchar = out[ out_pos ];
								return out_pos;
							}
							fragment = (char)decode_value( *in_iter++ );
						}
						while ( fragment < 0 );
						out[ out_pos++ ] |= ( fragment & 0x01c ) >> 2;
						out[ out_pos ] = ( fragment & 0x003 ) << 6;
				case step_C:
						do {
							if ( in_iter == in.end() )
							{
								_state._step = step_C;
								_state.plainchar = out[ out_pos ];
								return out_pos;
							}
							fragment = (char)decode_value( *in_iter++ );
						}
						while ( fragment < 0 );
						out[ out_pos ] |= ( fragment << 1 );
				case step_D:
						do {
							if ( in_iter == in.end() )
							{
								_state._step = step_D;
								_state.plainchar = out[ out_pos ];
								return out_pos;
							}
							fragment = (char)decode_value( *in_iter++ );
						}
						while ( fragment < 0 );
						out[ out_pos++ ] |= ( fragment & 0x010 ) >> 4;
						out[ out_pos ] = ( fragment & 0x00f ) << 4;
				case step_E:
						do {
							if ( in_iter == in.end() )
							{
								_state._step = step_E;
								_state.plainchar = out[ out_pos ];
								return out_pos;
							}
							fragment = (char)decode_value( *in_iter++ );
						}
						while ( fragment < 0 );
						out[ out_pos++ ] |= ( fragment & 0x01e ) >> 1;
						out[ out_pos ] = ( fragment & 0x001 ) << 7;
				case step_F:
						do {
							if ( in_iter == in.end() )
							{
								_state._step = step_F;
								_state.plainchar = out[ out_pos ];
								return out_pos;
							}
							fragment = (char)decode_value( *in_iter++ );
						}
						while ( fragment < 0 );
						out[ out_pos ] |= fragment << 2;
				case step_G:
						do {
							if ( in_iter == in.end() )
							{
								_state._step = step_G;
								_state.plainchar = out[ out_pos ];
								return out_pos;
							}
							fragment = (char)decode_value( *in_iter++ );
						}
						while ( fragment < 0 );
						out[ out_pos++ ] |= ( fragment & 0x018 ) >> 3;
						out[ out_pos ] = ( fragment & 0x007 ) << 5;
				case step_H:
						do {
							if ( in_iter == in.end() )
							{
								_state._step = step_H;
								_state.plainchar = out[ out_pos ];
								return out_pos;
							}
							fragment = (char)decode_value( *in_iter++ );
						}
						while ( fragment < 0 );
						out[ out_pos++ ] |= fragment;
					;
					}
				}
				return out_pos;
			}

			int decode_value( char in )
			{
				int result = -1;
				if ( _alphabet_map.empty() )
				{
					build_map();
				}
				char_lookup_t::const_iterator found = _alphabet_map.find( in );
				if ( found != _alphabet_map.end() )
				{
					result = found->second;
				}
				return result;
			}

			void build_map()
			{
				char ch = 0;
				for ( std::string::const_iterator iter = _alphabet.begin();
					iter != _alphabet.end();
					++iter )
				{
					_alphabet_map[ *iter ] = ch++;
				}
			}
		private:
			state _state;
			const std::string& _alphabet;
			char_lookup_t _alphabet_map;
		}; // class base_32
	}; // namespace decode
}; // namespace tvr

#endif // _TVR_BASE_32_STOP_THE_INSANITY_H
