
/**
 * \mainpage
 * The basic_substring class provides generic substring capabilities to the standard string class, as
 * recommended in Section 20.3.13 in "The C++ Programming Language" by Stroustrup.  But, not being
 * content to just duplicate what I saw there, I had to extend it a little bit.
 **/

/**
 * \file basic_substring.h
 * Provides everything needed to work with basic_substring, substring, and wsubstring classes.
 **/

// Protect from double-inclusion...
#ifndef basic_substring_h_sdlksldkslkdwelkjsodiuslkeslkduoileiousd
#define basic_substring_h_sdlksldkslkdwelkjsodiuslkeslkduoileiousd

#include <string>

/**
 * Used for extensions to the std created by myself.
 **/

namespace tvr_std {

/**
 * Provides generic substring capabilities to basic_string.
 *
 * This is inspired by Stroustrup's example in Section 20.3.13
 * of his "The C++ Programming Language".  In it, he describes
 * a substring class that's a little lighter than this one.
 * He also mentioned that it might be more interesting to create
 * a regular expression class along the same line (perhaps a
 * project for another day).  I thought it might be useful,
 * so I extended it.
 *
 * Basically, use the constructors to create a new substring.  If
 * you add a 'true' to the list of parameters, the substring will
 * work in reverse (starting at the end, and working its way towards
 * the beginning).
 *
 * You may iterate to the next matching substring using either of
 * the iterator operators (post or prefix, decrement or increment).
 * The iterators will not modify the actual string.
 *
 * The only way to actually modify a string through the substring
 * is to assign a string to the substring.  This causes the substring
 * to replace the substring in the string with whatever you assigned,
 * and iterate the substring to the next matching substring.
 *
 * When you iterate past the end, the substring becomes invalid, and
 * is no longer useful.  You may, however, assign another substring
 * to it (assigning substrings to a substring acts as a copy, and
 * does not modify the string).
 *
 * You may use the string operator to acquire the text of the substring.
 *
 * You may also use the bool operator to test whether the substring
 * is valid or not.
 *
 * Some examples:
 *
 * @code
 * #include "basic_substring.h"
 *
 * using namespace std;
 * using namespace tvr_std;
 *
 * string someText("I'm a little teapot, short and stout.");
 * cout << someText << endl;
 * substring sub(someText, "little");
 * sub = "gaudy";
 * sub = substring(someText, "short");
 * sub = "gaudy";
 * sub = substring(someText, "stout");
 * sub = "gaudy";
 * // By now, the string is "I'm a gaudy teapot, gaudy and gaudy.".
 * cout << someText << endl;
 * sub = substring(someText, "gaudy");
 * while (sub) {
 *	 // uses the bool operator to check for validity
 *	 sub = "little";
 * }
 * // Now, the string is "I'm a little teapot, little and little."
 * cout << someText << endl;
 * sub = substring(someText, "little", true); // reverse substring.
 * sub = "stout";
 * sub = "short";
 * // And so, we should be back to "I'm a little teapot, short and stout."
 * cout << someText << endl;
 * @endcode
 *
 **/

template<class Ch>
class basic_substring {
public:
    /// The string's size_type.
	typedef typename std::basic_string<Ch>::size_type size_type;

	/**
	 * @name Constructors.
	 **/
	//@{
	/**
	 * Manual constructor.
	 * Lets you manually set up the substring.
	 * @param s The string whose substring you want.
	 * @param start The starting position for the substring.
	 * @param len The length of the substring.
	 * @param doReverse When set to true, this substring iterates in reverse.
  	 **/
	basic_substring(std::basic_string<Ch>& s,
				   	size_type start,
				   	size_type len,
				   	bool doReverse=false) : ps(&s), n(len), pos(start) {
		if ((pos + n) > s.length()) {
			n = s.length();
		}
		this->doReverse = doReverse;
	};

	/**
	 * String constructor.
	 * Lets you create a substring based on another string.
	 * @param s The string whose substring you want.
	 * @param s2 The substring you want to find.
	 * @param doReverse When set to true, this substring iterates in reverse.
	 **/
	basic_substring(std::basic_string<Ch>& s,
				   	const std::basic_string<Ch>& s2,
				   	bool doReverse=false) : ps(&s), n(s2.length()) {
		if (doReverse)
			pos = s.rfind(s2);
		else
			pos = s.find(s2);
		this->doReverse = doReverse;
	};

	/**
	 * Character string constructor.
	 * Lets you create a substring based on the base character string type.
	 * @param s The string whose substring you want.
	 * @param p The substring you want to find.
	 * @param doReverse When set to true, this substring iterates in reverse.
	 **/
	basic_substring(std::basic_string<Ch>& s,
				   	const Ch* p,
				   	bool doReverse=false) : ps(&s) {
		std::basic_string<Ch> temp(p);
		n = temp.length();
		if (doReverse)
			pos = s.rfind(temp);
		else
			pos = s.find(temp);
		this->doReverse = doReverse;
	};
	//@}

	/**
	 * @name Assignment operators.
	 *
	 * The assignment operators will modify the original string this substring
	 * was constructed from.  The substring will attempt to find the next substring
	 * matching the original substring.
	 * 
	 * If this substring is invalid, nothing happens (not even a thrown exception).
	 * This runs counter to the original idea in Stroustrup's book.  I view it as an
	 * enhancement.  Avoid infinite loops with the 'bool' operator.
	 *
	 **/

	//@{

	/**
	 * Assigns a string to the substring.
	 * This replaces the contents of the substring with a new string,
	 * iterating to the next substring.
	 * @param in The string you want to use to replace the substring.
	 * @return Returns this.
	 **/
	basic_substring& operator=(const std::basic_string<Ch>& in) {
		if ((*this) == true) {
			std::basic_string<Ch> s = (*this);
			ps->replace(pos, n, in);
			if (doReverse)
				pos = ps->rfind(s, pos-1);
			else
				pos = ps->find(s, pos+1);
		}
		return *this;
	};

	/**
	 * Assigns a substring to this substring.
	 * This completely changes the original state of the substring.  This doesn't
	 * merely take the incoming substring's 'string' and assign it to this substring,
	 * this changes the internal state to match that of the incoming string.  Use this
	 * operator to reuse substring variables.
	 * @param in The substring you want to copy.
	 * @return Returns this.
	 **/
	basic_substring& operator=(const basic_substring<Ch>& in) {
		ps = in.ps;
		pos = in.pos;
		n = in.n;
		doReverse = in.doReverse;
		return *this;
	};

	/**
	 * Assigns a set of characters to the substring.
	 * This replaces the contents of the substring with a new string,
	 * iterating to the next substring.
	 * @param in The string you want to use to replace the substring.
	 * @return Returns this.
	 **/
	basic_substring& operator=(const Ch* in) {
		std::basic_string<Ch> s(in);
		return (*this) = s;
	};

	/**
	 * Assigns a character to the substring.
	 * This replaces the contents of the substring with a single character.
	 * This iterates to the next substring thereafter.
	 * @param in The character you want to use to replace the substring.
	 * @return Returns this.
	 **/
	basic_substring& operator=(Ch in) {
		std::basic_string<Ch> s;
		s = in;
		return (*this) = s;
	};
	//@}

	/**
	 * @name Inserts text before or after the substring.
	 * This does <b>not</b> advance the substring.  Use the iterators afterwards.
	 **/
	//@{

	/**
	 * Inserts the text of the string before the substring.
	 * @param in The string you wish to prepend.
	 * @return Returns this.
	 **/
	basic_substring& prepend(const std::basic_string<Ch>& in) {
		ps->insert(pos, in);
		pos += in.length();
		return *this;
	};

	/**
	 * Inserts the text of the string before the substring.
	 * @param in The string you wish to prepend.
	 * @return Returns this.
	 **/
	basic_substring& prepend(const Ch* in) {
		std::basic_string<Ch> arg(in);
		return prepend(arg);
	};

	/**
	 * Inserts the text of the string after the substring.
	 * @param in The string you wish to append.
	 * @return Returns this.
	 **/
	basic_substring& append(const std::basic_string<Ch>& in) {
		size_type tmp = pos + n;
		ps->insert(tmp, in);
		return *this;
	};

	/**
	 * Inserts the text of the string after the substring.
	 * @param in The string you wish to append.
	 * @return Returns this.
	 **/
	basic_substring& append(const Ch* in) {
		std::basic_string<Ch> arg(in);
		return append(arg);
	};
	//@}

	/**
	 * @name Casting operators.
	 **/

	//@{

	/**
	 * Acquires the text of the substring.
	 * In case you forgot.
	 **/
	operator const std::basic_string<Ch> () const {
		return std::basic_string<Ch> (*ps, pos, n);
	};
	/**
	 * Acquires the state of the substring.
	 * @return Returns true if the substring is valid, false otherwise.
	 **/
	operator const bool () const {
		return pos != std::basic_string<Ch>::npos;
	};
	//@}

	/**
	 * @name Iterator operators.
	 * When the iterator operator reaches the end, the
	 * substring becomes invalid.  Once a substring becomes
	 * invalid, it will always remain invalid.  You cannot
	 * reverse-iterate back to a valid substring.
	 **/

	//@{

	basic_substring& operator ++() {
		std::basic_string<Ch> substring;
		substring = (std::basic_string<Ch>)*this;
		pos = ps->find(substring, pos+1);
		return *this;
	};

	basic_substring& operator ++(int) {
		return ++(*this);
	};
	
	basic_substring& operator --() {
		std::basic_string<Ch> substring;
		substring = (std::basic_string<Ch>)*this;
		if (pos != 0) {
			pos = ps->rfind(substring, pos-1);
		} else {
			pos = size_type::npos;
		}
		return *this;
	};

	basic_substring& operator --(int) {
		return --(*this);
	};
	//@}
private:
	std::basic_string<Ch>* ps;
	size_type pos;
	size_type n;
	bool doReverse;
};

/// For use with 'string' typedef...
typedef basic_substring<char> substring;
/// For use with 'wstring' typedef...
typedef basic_substring<wchar_t> wsubstring;

}; // namespace tvr_std

#endif // basic_substring_h_sdlksldkslkdwelkjsodiuslkeslkduoileiousd
