// ArchiveEntry implementation

#include "ArchiveEntry.h"
#include <sys/stat.h>
#include <unistd.h>
#include <fstream>
#include <iostream>
#include "error.h"
#include "misc.h"

using namespace std;

string ArchiveEntry::prevClassID;
string ArchiveEntry::nextClassID;
string ArchiveEntry::upClassID;
set<ArchiveEntry> listItems;
string _ArchiveEntryPath;

const bool operator==(const ArchiveEntry &a, const ArchiveEntry &b) {
	return a.fileName == b.fileName;
}

const bool operator<(const ArchiveEntry &a, const ArchiveEntry &b) {
	return a.fileName < b.fileName;
}

istream &operator>>(istream &input, ArchiveEntry &a) {
	char ignore[10]; // This exists to help make things pretty.
	ios::fmtflags flags = input.flags();
	flags = input.unsetf(ios::skipws);
	GetString(input, a.fileName);
	GetString(input, a.order);
	GetString(input, a.title);
	GetString(input, a.prevLink);
	GetString(input, a.nextLink);
	GetString(input, a.upLink);
	input >> a.m_time;
	input.getline(ignore, 10); // there, we throw it away, 'cause we don't care.
	input.flags(flags);
	return input;
}

ostream &operator<<(ostream &output, const ArchiveEntry &a) {
	SetString(output, a.fileName);
	SetString(output, a.order);
	SetString(output, a.title);
	SetString(output, a.prevLink);
	SetString(output, a.nextLink);
	SetString(output, a.upLink);
	output << a.m_time << endl; // we add a new line to make things pretty.
	return output;
}

ArchiveEntry::ArchiveEntry() {
}

ArchiveEntry::ArchiveEntry(const string &aFileName) {
	// Strategy:
	//
	// Hunt through the file, acquiring as much information as we may.
	// First, let's see if we can find it in our list.
	fileName = aFileName;
	struct stat fileInfo;
	stat(fileName.c_str(), &fileInfo);
	m_time = fileInfo.st_mtime;
	ArchiveEntry test;
	test.fileName = aFileName;
	set<ArchiveEntry>::iterator found = listItems.find(test);
	if (found != listItems.end())
		test = *found;
	if (found != listItems.end() && (m_time <= test.m_time)) {
		*this = *found;
		return; // We're done.  It's already cached.
	}
	// Now, let's suck the file info in so we can look at it.
	ifstream inFile(fileName.c_str());
	if (!inFile) {
		cerr << "Unable to open " << fileName << " for reading." << endl;
		return;
	}
	inFile.unsetf(ios::skipws); // we want white space ... 
	string fileText;
	while (!inFile.eof()) {
		char ch;
		inFile >> ch;
		fileText += ch;
	}
	// Now we'll find our attributes.
	SetOrder(fileText);
	SetTitle(fileText);
	SetPrevLink(fileText);
	SetNextLink(fileText);
	SetUpLink(fileText);
	listItems.erase(*this);
	listItems.insert(*this);
}

ArchiveEntry::ArchiveEntry(const ArchiveEntry &origEntry) {
	*this = origEntry;
}

ArchiveEntry::~ArchiveEntry() {
}

void ArchiveEntry::GetEntryVars(map<string, string> &entryVars) {
	// Strategy:
	// Add any variables we need to entryVars.
	// Two I can immediately think of are "fileName" and "title".
	entryVars["FILENAME"] = fileName;
	entryVars["TITLE"] = title;
	entryVars["ORDER"] = order;
}

void ArchiveEntry::operator=(const ArchiveEntry &origEntry) {
	fileName = origEntry.fileName;
	order = origEntry.order;
	title = origEntry.title;
	prevLink = origEntry.prevLink;
	nextLink = origEntry.nextLink;
	upLink = origEntry.upLink;
	begin = origEntry.begin;
	end = origEntry.end;
	m_time = origEntry.m_time;
}

void ArchiveEntry::SetOrder(const string &inVec) {
	// Strategy:
	//
	// Hunt through the vector and find the <archive> tag.
	// When you find it, use HTMLTag to help get the attribute
	// we want.
	string archiveTag("<archive");
	string test = inVec;
	string::iterator start = search(test.begin(), test.end(), archiveTag.begin(), archiveTag.end(), no_case);
	if (start == test.end()) {
		return; // No archive tag.
	}
	HTMLTag archive(start, test.end());
	string foundOrder = archive.GetAttribute("order").GetValue();
	if (no_case_compare(foundOrder, order)) {
		return; // nothing to do.
	}
	order = foundOrder;
}

void ArchiveEntry::SetTitle(const string &inVec) {
	// Strategy:
	// Want to put any text between <title> and </title> into 'title'.
	string titleTag("<title>");
	string endTitle("</title>");
	string test = inVec;
	string::iterator start = search(test.begin(), test.end(), titleTag.begin(), titleTag.end(), no_case);
	if (start == test.end()) {
		// Nothing to do.  Doesn't have a <title>
		return;
	}
	start = start + 7;
	string::iterator end = search(start, test.end(), endTitle.begin(), endTitle.end(), no_case);
	string newTitle = string(start, end);
	if (no_case_compare(newTitle, title)) {
		return; // Nothing to do.
	}
	title = newTitle;
}

const string GetClassText(const string &inVec, const string &classID) {
	// Here's a challenge...
	//
	// Need to hunt through this and find any tag that has the
	// class ID specified earlier for classID.  Then, anything between
	// that tag and its ending tag are returned for output.
	string output;
	if (classID.length()==0) return output; // nothing to do.
	string testVector = inVec;
	string::iterator next = testVector.begin();
	string::iterator foundBegin;
	string tagName;
	int count = 0;
	while (next != testVector.end()) {
		if (*next == '<') {
			// Found the start of a tag.
			// Let's see if it has our class id.
			HTMLTag tag(next, testVector.end());
			SAFEHTMLSTART()
			if (no_case_compare(tag.GetAttribute("class").GetValue(), classID)) {
				// Found a match!  Now we need to find the end of this.
				tagName = tag.GetTag();
				foundBegin = tag.end();
				next = foundBegin;
				++count;
				break;
			}
			SAFEHTMLSTOP()
			next = tag.end();
		}
		++next;
	}
	if (next == testVector.end()) {
		// Couldn't find it.  It probably doesn't exist.
		return output;
	}
	string endTagName = "/" + tagName;
	while (next != testVector.end()) {
		if (*next == '<') {
			// Found the start of a tag.
			// Let's see if it's the end tag.
			HTMLTag tag(next, testVector.end());
			if (no_case_compare(tag.GetTag(), tagName)) {
				++count; // keep everything balanced.
			} else if (no_case_compare(tag.GetTag(), endTagName)) {
				--count; // keep everything balanced.
				if (count==0) {
					// We're done.  We've found what we're looking for.
					output = string(foundBegin+1, next);
					return output;
				}
			}
			next = tag.end();
		}
		++next;
	}
	return output; // give 'em whatever we've got.
}

void ArchiveEntry::SetPrevLink(const string &inVec) {
	if (prevClassID.length()==0) return;
	prevLink = GetClassText(inVec, prevClassID);
}

void ArchiveEntry::SetNextLink(const string &inVec) {
	if (nextClassID.length()==0) return; // nothing to do.
	nextLink = GetClassText(inVec, nextClassID);
}

void ArchiveEntry::SetUpLink(const string &inVec) {
	if (upClassID.length()==0) return; // nothing to do.
	upLink = GetClassText(inVec, upClassID);
}

const string ArchiveEntry::GetText() {
	return fileName;  // for now.. must modify this to do what it should.
}

const bool ArchiveEntry::operator==(const string &aFileName) {
	return fileName == aFileName;
}

void ArchiveEntry::SetPrefFile(const string &inPrefFile) {
	// Strategy:
	// Stream the contents of listItems to inPrefFile.
	set<ArchiveEntry>::iterator next = listItems.begin();
	if (listItems.size()) {
		ofstream outFile(inPrefFile.c_str());
		if (!outFile) {
			cerr << "Note: unable to write preference file [" << inPrefFile << "]." << endl;
			return;
		}
		ArchiveEntry holding;
		while (next != listItems.end()) {
			holding = *next;
			if (holding.fileName.length()) {
				outFile << holding;
			}
			++next;
		}
		outFile << flush;
	} // otherwise, we don't have anything to do.
}

void ArchiveEntry::GetPrefFile(const string &inPrefFile) {
	// Strategy:
	// Stream the contents of inPrefFile to listItems.
	ifstream inFile(inPrefFile.c_str(), ios::in | ios::binary);
	if (!inFile) {
		cerr << "Note: unable to read preference file [" << inPrefFile << "]." << endl;
		return;
	}
	listItems.clear();
	while (!inFile.eof()) {
		ArchiveEntry newEntry;
		inFile >> newEntry;
		listItems.insert(newEntry);
	}
}

const string ArchiveEntry::GetLatest(const string &latestID) {
	// Strategy:
	// Get all the text in the file, then find the first tag of class
	// 'latestID'.  Get all text between that tag and its ending tag.
	//
	// GetClassText will handle this for us.  We just have to read
	// the file in.
	ifstream iStr(fileName.c_str());
	if (!iStr) {
		throw ArchiveError(ArchiveError::eNoOpenEntry);
	}
	iStr.unsetf(ios::skipws); // we like whitespace.
	string entryString;
	char ch;
	while (!iStr.eof()) {
		iStr >> ch;
		entryString += ch;
	}
	return GetClassText(entryString, latestID);
}

int sortByTitle(const void *a, const void *b) {
	int result = 0;
	const dirent *a1 = *(dirent**)a;
	const dirent *b1 = *(dirent**)b;
	// The constructor caches... so this probably won't cost much time.
	string aName = _ArchiveEntryPath + a1->d_name;
	string bName = _ArchiveEntryPath + b1->d_name;
	ArchiveEntry aEntry(aName);
	ArchiveEntry bEntry(bName);
	result = (aEntry.title < bEntry.title) ? -1 : 1 ;
	if (reverseSort) 
		return (result * -1);
	return result;
}

int sortByOrderTag(const void *a, const void *b) {
	int result = 0;
	const dirent *a1 = *(dirent**)a;
	const dirent *b1 = *(dirent**)b;
	// The constructor caches... so this probably won't cost much time.
	string aName = _ArchiveEntryPath + a1->d_name;
	string bName = _ArchiveEntryPath + b1->d_name;
	ArchiveEntry aEntry(aName);
	ArchiveEntry bEntry(bName);
	result = (aEntry.order < bEntry.order) ? -1 : 1 ;
	if (reverseSort) 
		return (result * -1);
	return result;
}

// vim:ai ts=4
