// ArchiveTag implementation

#include "ArchiveTag.h"
#include "ArchiveEntry.h"
#include "misc.h"
#include "tag.h"
#include <fstream>
#include <iostream>
#include <string>
#include <math.h>
#include <stdio.h>
#include "error.h"

ArchiveTag::ArchiveTag(HTMLTag &tag) {
	*this = tag;
}

void ArchiveTag::DoForeExec() {
	if (!foreExec.empty()) {
		int result = system(foreExec.c_str());
		if (result) {
			cerr << "NOTE: [" << foreExec << "] returned " << result << "." << endl;
		}
	}
	if (nextTag) {
		nextTag->DoForeExec();
	}
}

void ArchiveTag::DoPostExec() {
	if (!postExec.empty()) {
		int result = system(postExec.c_str());
		if (result) {
			cerr << "NOTE: [" << postExec << "] returned " << result << "." << endl;
		}
	}
	if (nextTag) {
		nextTag->DoPostExec();
	}
}

void ArchiveTag::WriteFile(const string &output) {
	DoForeExec();
	if (outboundFileName.empty()) {
		if (nextTag) {
			nextTag->WriteFile(output);
		} else if (isIndex || latestID.length()) {
			throw ArchiveError(ArchiveError::eNoOutboundFile);
		}
		return;
	}
	string myFile;
	string myPath;
	splitFilePath(outboundFileName, myFile, myPath);
	string filePath = outboundPath.cd(myPath);
	filePath += myFile;
	ofstream oStr(filePath.c_str());
	if (!oStr) {
		throw ArchiveError(ArchiveError::eNoOpenOutboundFile);
		return;
	}
	cerr << "Replaced: [" << filePath << "]." << endl;
	oStr << output;
	DoPostExec();
}

string ArchiveTag::GetLatest() {
	// Strategy:
	// Find the latest item in the archiveEntries vector, and
	// call its GetLatest method.
	string output;
	if (latestID.empty())
		return output; // nothing to return.
	if (archiveEntries.empty())
		return output; // Again, nothing to return.
	// Figure out the latest item.
	ArchiveEntry &latestEntry = archiveEntries[0]; // yup.. reversed.
	cerr << "Latest: " << latestEntry.fileName << endl;
	return latestEntry.GetLatest(latestID);
}

void ArchiveTag::GetXY(unsigned long int &x, unsigned long int &y) {
	int numEntries = archiveEntries.size();
	// we need to determine how many rows and columns we'll need, so we can
	// build a table if we want to.
	if (numEntries == 1) {
		x = 1;
		y = 1;
	} else {
		if (numEntries > maxCol) {
			bool found = false;
			double height;
			for (size_t width = maxCol; width >= minCol && !found; width--) {
				// We seek a 'box' with no dangling entries at the bottom.
				// To that end, we seek a factor from maxCol to minCol
				// inclusive, of numEntries.  We prefer larger values to
				// smaller ones.
				double test = modf((float)numEntries/(float)width, &height);
				if (test == 0.0) {
					// We've found the values.
					x = width;
					y = (unsigned long int) height;
					found = true;
				}
			}
			if (!found) {
				// We couldn't find the aforementioned factor. Gonna have to
				// have holes.
				// A more sophisticated approach would give us the most entries
				// with the fewest holes.
				modf((float)numEntries/(float)maxCol, &height);
				x = maxCol;
				y = (unsigned int) height+1;
			}
		} else {
			// We do not have enough entries to worry about making columns.
			x = numEntries;
			y = 1;
		}
	}	
}
string ArchiveTag::GetIndex() {
	string entries;
	size_t numEntries = archiveEntries.size();
	unsigned long int x = 0, y = 0;
	GetXY(x, y);
	size_t curPos = 0;
	// foreList...
	// First, variables.
	map<string, string> listVars = GetListVars();
	string output = substituteVars(listVars, foreList);
	entries += output;
	for (unsigned long int yy = 0; (yy < y) && (curPos < numEntries); yy++) {
		// foreGroup
		map<string, string> groupVars = GetGroupVars();
		output = substituteVars(groupVars, foreGroup);
		entries += output;
		for (unsigned long int xx = 0; (xx < x) && (curPos < numEntries); xx++) {
			ArchiveEntry &curEntry = archiveEntries[curPos++];
			// foreEntry
			map<string, string>entryVars = GetEntryVars();
			curEntry.GetEntryVars(entryVars);
			output = substituteVars(entryVars, foreEntry);
			entries += output;
			// Entry
			output = substituteVars(entryVars, entryText);
			entries += output;
			// postEntry
			output = substituteVars(entryVars, postEntry);
			entries += output;
		}
		// postGroup
		output = substituteVars(groupVars, postGroup);
		entries += output;
	}
	// postList...
	output = substituteVars(listVars, postList);
	entries += output;
	return entries;
}

void ArchiveTag::operator=(HTMLTag &tag) {
	// Hunt through the tag, finding all the happy little attributes
	// and acquiring their values.
	init();

	reverseSort = false; // from misc.h

	SAFEHTMLSTART()
	tag.GetAttribute(string("index"));
	isIndex = true;
	SAFEHTMLSTOP()

	SAFEHTMLSTART()
	tag.GetAttribute(string("reverse"));
	reverseSort = true;
	SAFEHTMLSTOP()

	SAFEHTMLSTART()
	outboundFileName = tag.GetAttribute(string("outbound")).GetValue();
	SAFEHTMLSTOP()

	SAFEHTMLSTART()
	latestID = tag.GetAttribute(string("latest")).GetValue();
	SAFEHTMLSTOP()

	SAFEHTMLSTART()
	prevLink = tag.GetAttribute(string("prevLink")).GetValue();
	SAFEHTMLSTOP()

	SAFEHTMLSTART()
	prevLinkClass = tag.GetAttribute(string("prevLinkClass")).GetValue();
	SAFEHTMLSTOP()

	SAFEHTMLSTART()
	nextLink = tag.GetAttribute(string("nextLink")).GetValue();
	SAFEHTMLSTOP()

	SAFEHTMLSTART()
	nextLinkClass = tag.GetAttribute(string("nextLinkClass")).GetValue();
	SAFEHTMLSTOP()

	SAFEHTMLSTART()
	upLinkClass = tag.GetAttribute(string("upLinkClass")).GetValue();
	SAFEHTMLSTOP()

	SAFEHTMLSTART()
	noPrevious = tag.GetAttribute(string("noPrevious")).GetValue();
	SAFEHTMLSTOP()

	SAFEHTMLSTART()
	noNext = tag.GetAttribute(string("noNext")).GetValue();
	SAFEHTMLSTOP()

	SAFEHTMLSTART()
	foreExec = tag.GetAttribute(string("foreExec")).GetValue();
	SAFEHTMLSTOP()

	SAFEHTMLSTART()
	postExec = tag.GetAttribute(string("postExec")).GetValue();
	SAFEHTMLSTOP()

	SAFEHTMLSTART()
	nextIndex = tag.GetAttribute(string("nextIndex")).GetValue();
	SAFEHTMLSTOP()

	SAFEHTMLSTART()
	string tmpString = tag.GetAttribute(string("regexp")).GetValue();
	wildCard = new char [tmpString.length()]; // note: from misc.h
	strcpy(wildCard, tmpString.c_str());
	SAFEHTMLSTOP()

	// Columnar vars...

	SAFEHTMLSTART()
	determineSortType(tag.GetAttribute("sortOrder").GetValue());
	SAFEHTMLSTOP()

	SAFEHTMLSTART()
	foreList = tag.GetAttribute(string("foreList")).GetValue();
	SAFEHTMLSTOP()

	SAFEHTMLSTART()
	postList = tag.GetAttribute(string("postList")).GetValue();
	SAFEHTMLSTOP()

	SAFEHTMLSTART()
	foreGroup = tag.GetAttribute(string("foreGroup")).GetValue();
	SAFEHTMLSTOP()

	SAFEHTMLSTART()
	postGroup = tag.GetAttribute(string("postGroup")).GetValue();
	SAFEHTMLSTOP()

	SAFEHTMLSTART()
	foreEntry = tag.GetAttribute(string("foreEntry")).GetValue();
	SAFEHTMLSTOP()

	SAFEHTMLSTART()
	postEntry = tag.GetAttribute(string("postEntry")).GetValue();
	SAFEHTMLSTOP()

	SAFEHTMLSTART()
	entryText = tag.GetAttribute(string("entryText")).GetValue();
	SAFEHTMLSTOP()

	maxCol = 0;
	minCol = 0;
	SAFEHTMLSTART()
	string maxColString = tag.GetAttribute(string("maxCol")).GetValue();
	maxCol = strtoul(maxColString.c_str(), NULL, 10);
	SAFEHTMLSTOP()

	SAFEHTMLSTART()
	string minColString = tag.GetAttribute(string("minCol")).GetValue();
	minCol = strtoul(minColString.c_str(), NULL, 10);
	SAFEHTMLSTOP()

	SAFEHTMLSTART()
	entryPath = Dir(tag.GetAttribute("entryPath").GetValue());
	SAFEHTMLSTOP()

	SAFEHTMLSTART()
	outboundPath = Dir(tag.GetAttribute("outboundPath").GetValue());
	SAFEHTMLSTOP()

	createArchiveEntries();
}

void ArchiveTag::determineSortType(const string &sortString) {
	// Strategy:
	// Compare the incoming string with our strings, and set 'sortType'
	// accordingly.  Pretty simple.
	if (no_case_compare(sortString, "modificationdate")) {
		sortType = eModificationDate;
	} else if (no_case_compare(sortString, "creationdate")) {
		sortType = eCreationDate;
	} else if (no_case_compare(sortString, "filename")) {
		sortType = eFilename;
	} else if (no_case_compare(sortString, "ordertag")) {
		sortType = eOrderTag;
	} else if (no_case_compare(sortString, "title")) {
		sortType = eTitle;
	} else {
		// We'll default to eFilename.
		sortType = eFilename;
	}
}

void ArchiveTag::createArchiveEntries() {
	// Strategy:
	//
	// Use scandir with select and the various sortByXXX methods
	// to get an array of files.  Use this array of files to build
	// the archiveEntries list.
	//
	// If we have no text in wildCard, just return without doing anything.
	if (wildCard == 0) // from misc.h
		return;
	if (strlen(wildCard)==0) // from misc.h
		return;
	LPCOMPAREFN compare = 0;
	switch (sortType) {
	case eModificationDate:
		{
			compare = sortByModDate;
		}
		break;
	case eCreationDate:
		{
			compare = sortByCreDate;
		}
		break;
	case eFilename:
		{
			compare = sortByFilename;
		}
		break;
	case eOrderTag:
		{
			compare = sortByOrderTag;
		}
		break;
	case eTitle:
		{
			compare = sortByTitle;
		}
		break;
	}
	_ArchiveEntryPath = entryPath.GetPathRelativeTo(Dir::getcwd());
	dirent **nameList;
	int numEntries = scandir(_ArchiveEntryPath.c_str(), &nameList, select, compare);
	for (int i = 0; i < numEntries; i++) {
		string fname(nameList[i]->d_name);
		if (_ArchiveEntryPath.length()) {
			_ArchiveEntryPath.c_str();
			fname = _ArchiveEntryPath + fname;
		}
		cerr << "Found file: [" << fname << "]" << endl;
		ArchiveEntry foundEntry(fname); // automatically puts it in the list or extracts from the list as needed.
		archiveEntries.push_back(foundEntry);
	}
	if (numEntries == 0 ) {
		cerr << "Unable to find files." << endl;
	}
}

void ArchiveTag::init() {
	isIndex = false;
	reverseSort = false;
	nextTag = 0;
}

map<string, string> ArchiveTag::GetListVars() {
	const char *ud = "%u";
	map<string, string> vars;
	unsigned long int x = 0, y = 0;
	char stringX[50], stringY[50];
	GetXY(x, y);
	sprintf(stringX, ud, x);
	sprintf(stringY, ud, y);
	vars["X"] = stringX;
	vars["Y"] = stringY;
	char totalEntries[50];
	sprintf(totalEntries, ud, archiveEntries.size());
	vars["TOTALENTRIES"] = totalEntries;
	return vars;
}

void ArchiveTag::CreateForeList(string &output) {
	map<string, string> vars = GetListVars();
}

void ArchiveTag::CreatePostList(string &output) {
	map<string, string> vars = GetListVars();
}

map<string, string> ArchiveTag::GetGroupVars() {
	map<string, string> vars;
	return vars;
}

void ArchiveTag::CreateForeGroup(string &output) {
	map<string, string> vars = GetGroupVars();
}

void ArchiveTag::CreatePostGroup(string &output) {
	map<string, string> vars = GetGroupVars();
}

map<string, string> ArchiveTag::GetEntryVars() {
	map<string, string> vars;
	return vars;
}

void ArchiveTag::CreateForeEntry(string &output) {
	map<string, string> vars = GetEntryVars();
}

void ArchiveTag::CreatePostEntry(string &output) {
	map<string, string> vars = GetEntryVars();
}

// vim:ai ts=4
