#include "Dir.h"
#include <unistd.h>
#include <stdlib.h>
#include <algorithm>

Dir *Dir::virtualCWD = 0;

const string Dir::cd(const string &aPath) {
	if (aPath.empty()) {
		// Nothing to do.
		return (string)*this;
	}
	string testPath = aPath;
	string::size_type beginTestPath = 0;
	string::size_type endTestPath = testPath.length()-1;
	if (testPath[0]=='/') {
		// need to do some adjustin'.
		// We want the path to *not* start with the / character.
		path = "/";
		testPath = testPath.substr(beginTestPath+1, endTestPath-(beginTestPath));
	}
	size_t depth = 0;
	for (int i = 0; i < testPath.length(); i++) {
		if (testPath[i]=='/')
			++depth;
	}
	if ((depth == 1) && (testPath[endTestPath]=='/')) {
		// path ends with '/', but otherwise has no depth.
		testPath = testPath.substr(beginTestPath, (endTestPath+1)-(beginTestPath));
		depth = 0;
	}
	if (depth > 0) {
		// Handle this path one part at a time.
		string::size_type pos = beginTestPath;
		string::size_type next = endTestPath;
		while (true) {
			next = testPath.find('/', pos);
			if (next == string::npos) {
				// we're most likely done.
				next = testPath.length();
				cd(testPath.substr(pos, next-pos));
				break;
			}
			cd(testPath.substr(pos, next-pos));
			pos = next+1;
		}
		return (string)*this;
	}
	bool handled = false;
	if (testPath == "..") {
		// go up one directory.
		string::iterator end = path.end()-1;
		while (end != path.begin()) {
			--end;
			if (*end == '/') {
				break;
			}
		}
		if (end == path.begin())
			path = "./";
		else
			path = string(path.begin(), end+1); // we want to keep the '/' char.
		handled = true;
	}
	if (testPath == ".") {
		handled = true;
	}
	if (testPath == "~") {
		// Determine home, then go there.
		path = GetHome();
		handled = true;
	}
	if (!handled) {
		// Just tack whatever we have onto the path.
		path += testPath;
		path += "/";
	}
	return (string)*this;
}

const string Dir::GetPathRelativeFrom(const string &absPath) const {
	Dir test(absPath);
	return GetPathRelativeFrom(test);
}

const string Dir::GetPathRelativeFrom(const Dir &absPath) const {
	return absPath.GetPathRelativeTo(*this);
}

const string Dir::GetPathRelativeTo(const string &absPath) const {
	// User wants a relative path.. one that gets from *this to
	// absPath.
	return GetPathRelativeTo(Dir(absPath)); // just in case absPath isn't absolute.
}

const string Dir::GetPathRelativeTo(const Dir &absPath) const {
	if ((absPath.path == path) || (&absPath == this)) {
		// Same directory.
		return "./";
	}
	// Something more elaborate from this point on.
	const string inPath = absPath.path;
	size_t index = 0;
	size_t len = (inPath.length() < path.length()) ? inPath.length() : path.length();
	while (index < len) {
		if (inPath[index]!=path[index]) {
			// We've found a difference.
			// Now we need to back up to the previous '/' character, copy
			// both sides of the path into other strings, and
			// figure out how many '..'s it'll take to get to one side,
			// then append the new stuff to that many '..' sets.
			string::size_type lastSlash = inPath.rfind('/', index);
			string::size_type endInPath = inPath.length()-1;
			string leftSide = inPath.substr(0, lastSlash+1);
			const char *chLeftSide = leftSide.c_str();
			string rightSide = inPath.substr(lastSlash+1, (endInPath+1)-(lastSlash+1));
			if (rightSide[rightSide.length()-1]!='/') {
				rightSide += "/";
			}
			if (path == leftSide) {
				// The incoming path is our subdirectory.
				rightSide = "./" + rightSide;
				return rightSide;
			}
			Dir temp(path);
			while (true) {
				rightSide = "../" + rightSide;
				const char *chRightSide = rightSide.c_str();
				temp.cd("..");
				const char *chTempPath = temp.path.c_str();
				if (temp.path == leftSide) {
					return rightSide;
				}
			}
		}
		++index;
	}
	// The incoming path is a subdirectory.
	string::size_type pathLen = path.length();
	string::size_type inPathLen = inPath.length();
	string outpath = "./" + path.substr(inPathLen, pathLen-inPathLen);
	return outpath;
}

const string Dir::GetHome() {
	const char *test = getenv("HOME");
	if (test==0) {
		cerr << "This shell does not set $HOME." << endl;
		return string();
	}
	string output(test);
	if (output[output.length()-1]!='/')
		output += "/";
	return output;
}

const string Dir::GetAbsPath(const string &relPath) {
	// Presuming this is a relative path, resolve all '..', '~', '.'
	// things to an absolute path.
	Dir output;
	string::size_type next = 0;
	string::size_type end = relPath.length();
	const char ch = relPath[next];
	switch(ch) {
	case '~':
		// We start at home.
		// I guess the only way to get the absolute pathname for
		// home is to acquire the contents of the HOME environment
		// variable.
		output.path = Dir::GetHome();
		next += 2;
		break;
	case '.':
		output = Dir::getcwd();
		next += 2;
		if (relPath.length()>1 && relPath[1] == '.') {
			// We start one level above the current directory.
			output.cd("..");
			++next;
		}
		break;
	case '/':
		// We start at root.
		output.path = "/";
		++next;
		break;
	default :
		// We start at the current working directory.
		output = Dir::getcwd();
	}
	if (next < end) {
		string finish(relPath.substr(next, end-next));
		output.cd(finish);
	}
	return output;
}

Dir::Dir() {
	path = string(Dir::getcwd());
	(string) *this;
}

Dir::Dir(const string &newPath) {
	// Check newPath for relativity...
	if (isRelative(newPath)) {
		// need to determine what the absolute path would be.
		path = GetAbsPath(newPath);
	} else {
		path = newPath;
	}
	(string) *this;
}

bool Dir::isRelative(const string &inString) {
	// Check inString for relativity...
	if (inString.find("/.")!=string::npos) {
		return true;
	}
	if (inString.find("/..")!=string::npos) {
		return true;
	}
	if (inString.find("~")!=string::npos) {
		return true;
	}
	if (inString.find("./")!=string::npos) {
		return true;
	}
	if (inString.find("../")!=string::npos) {
		return true;
	}
	if (inString.find("~/")!=string::npos) {
		return true;
	}
	if (inString == ".") {
		return true;
	}
	if (inString == "..") {
		return true;
	}
	if (inString == "~") {
		return true;
	}
	return false;
}

void Dir::operator=(const Dir &inDir) {
	path = inDir.path;
}

Dir Dir::getcwd() {
	if (virtualCWD)
		return *virtualCWD;
#ifdef __USE_GNU
// using char *get_current_working_dir_name();
	char *charWD = get_current_dir_name();
	Dir outbound(charWD);
	free(charWD);
#else // __USE_GNU
// using char *getcwd(char *buf, size_t size);
	char *charWD = 0;
	size_t size = 100;
	// This isn't working on this machine.
	while (charWD == 0) {
		// POSIX.1 not extended on this machine.
		char *buf = new char[size];
		charWD = ::getcwd(buf, size);
		if (charWD) {
			// it's worked.. we're done.
			Dir outbound(charWD);
			delete buf;
			return outbound;
		}
		// Didn't work.  Try again with a larger size.
		size = size * 2;
		delete buf;
		buf = 0;
	}
	Dir outbound(charWD);
	delete charWD;
#endif // __USE_GNU
	if (outbound.path[outbound.path.length()-1]!='/') {
		outbound.path += "/";
	}
	return outbound;
}

Dir::operator string(){
	if (path[path.length()-1]!='/') {
		path += "/";
	}
	return path;
}

// vim:ai ts=4
