
#include "misc.h"
#ifdef HAVE_LIBREGEXX
#include <regexx.hh>
#else
#include <regex.h>
#endif
#include <sys/stat.h>
#include <algorithm>
#include <ctype.h>

#ifdef HAVE_LIBREGEXX
using namespace regexx;
#endif

char *wildCard = 0;
bool reverseSort = false;

int select(const dirent *entry) {
	// The following object is used to help deal with regular expressions.
	// This means, of course, that I'm doing everything Posix-compliant,
	// rather than the more limiting * and ? approach.
	//
	// It'll probably confuse folks.  Oh well.
#ifdef HAVE_LIBREGEXX
	// using the Regexx object.
	if (entry == 0) return 0; // obviously not a match.
	Regexx rxx; // http://projects.nn.com.br <gustavo@nn.com.br>
	try {
		if (rxx.exec(string(entry->d_name), string(wildCard))) {
			return 1;
		}
	} catch (Regexx::CompileException &e) {
		cerr << endl;
		cerr << e.message() << endl;
	}
	return 0;
#else // HAVE_LIBREGEXX
	// Handling things manually with regcomp/exec/error/free.
	regex_t preg;
	int result = regcomp(&preg, wildCard, REG_EXTENDED | REG_NOSUB);
	if (result) {
		// Uh oh.. didn't work.
		char errmsg[1024];
		size_t size = regerror(result, &preg, errmsg, 1024);
		cerr << "Regex error: " << errmsg << endl;
		regfree(&preg);
		return 0;
	}
	regmatch_t foo;
	result = regexec(&preg, entry->d_name, 0, &foo, 0);
	if (result) {
		if (result != REG_NOMATCH) {
			// Eh.. some kind of error.  Better tell the user.
			char errmsg[1024];
			size_t size = regerror(result, &preg, errmsg, 1024);
			cerr << "Regex error: " << errmsg << endl;
		}
		regfree(&preg);
		return 0;
	}
	regfree(&preg);
	return 1;
#endif // HAVE_LIBREGEXX
}

int sortByModDate(const void *a, const void *b) {
	int result = 0;
	struct stat aStat;
	struct stat bStat;
	const dirent *a1 = *(dirent**)a;
	const dirent *b1 = *(dirent**)b;
	stat(a1->d_name, &aStat);
	stat(b1->d_name, &bStat);
	if (aStat.st_mtime != bStat.st_mtime) {
		result = (aStat.st_mtime < bStat.st_mtime) ? -1 : 1 ;
	}
	if (reverseSort)
		return (result * -1);
	return result;
}

int sortByCreDate(const void *a, const void *b) {
	int result = 0;
	struct stat aStat;
	struct stat bStat;
	const dirent *a1 = *(dirent**)a;
	const dirent *b1 = *(dirent**)b;
	stat(a1->d_name, &aStat);
	stat(b1->d_name, &bStat);
	if (aStat.st_ctime != bStat.st_ctime) {
		result = (aStat.st_ctime < bStat.st_ctime) ? -1 : 1 ;
	}
	if (reverseSort)
		return (result * -1);
	return result;
}

int sortByFilename(const void *a, const void *b) {
	int result = 0;
	const dirent *a1 = *(dirent**)a;
	const dirent *b1 = *(dirent**)b;
	result = strcmp(a1->d_name, b1->d_name);
	if (reverseSort)
		return (result * -1);
	return result;
}
/*
int sortByOrderTag(const void *a, const void *b) {
	int result = 0;
	if (reverseSort)
		return (result * -1);
	return result;
}

int sortByOrderTitle(const void *a, const void *b) {
	int result = 0;
	if (reverseSort)
		return (result * -1);
	return result;
}
*/
ostream &SetString(ostream &to, const string &a) {
	// Strategy:
	// String to write must be wrapped in () characters,
	// and delimited with backslashes.
	to << '(';
	for (size_t i = 0; i < a.length(); i++) {
		char ch = a[i];
		if (ch == '\\' || ch == ')') {
			to << '\\';
		}
		to << ch;
	}
	to << ')';
	return to;
}

istream &GetString(istream &from, string &a) {
	// Strategy:
	// String is wrapped with () characters.
	// String is also delimited with backslashes.
	char ch;
	from.get(ch); // first one is ignored.
	while (!from.eof()) {
		from.get(ch);
		if (ch == ')')
			break;
		if (ch == '\\') {
			from.get(ch);
		}
		a += ch;
	}
	return from;
}


string substituteVars(const map<string, string> &vars, const string &inString) {
	map<string, string> myVars = vars;
	string output;
	string test(inString);
	string::iterator next = test.begin();
	string::iterator end = test.end();
	while (next != end) {
		if (*next == '%') {
			// Perhaps a variable.. perhaps not.
			if ((next+1 == end) || (*(next+1) == '%')) {
				// Not a variable.  Just a '%' symbol.
				output += *next;
				continue;
			} else {
				// Found a variable.
				++next;
				string::iterator varEnd = find(next, end, '%');
				string var = upperCase(string(next, varEnd));
				output += myVars[var];
				next = varEnd+1;
				continue;
			}
		}
		output += *next++;
	}
	return output;
}

const string upperCase(const string &lowerCaseString) {
	string output;
	char ch = '\0';
	for (size_t i = 0; i < lowerCaseString.length(); i++) {
		ch = lowerCaseString[i];
		ch = toupper(ch);
		output += ch;
	}
	return output;
}

void splitFilePath(const string &filePath, string &file, string &path) {
	string::size_type pos = filePath.find_last_of('/');
	if (pos!=string::npos) {
		path = filePath.substr(0, pos);
		file = filePath.substr(pos+1, filePath.length()-(pos+1));
	} else {
		path = string();
		file = filePath;
	}
}

// vim:ai ts=4
